/*
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved.

Purpose-
This file has the implementation of Image Editing functionality in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global $, liveViewObject, dwObject, liveExtensionsConfigPath*/

function DwExtensionEvent() {
    'use strict';
    this.listeners = {};
    this.contexts = {};
    this.isDebugging = true;
}

var DW_CONSTANTS = {
    EscapeKeyCode: 27,
    EnterKeyCode: 13,
    TabKeyCode: 9,
    SpaceKeyCode: 32,
    RightArrowKeyCode: 39,
    LeftArrowKeyCode: 37
};

var DW_EXTENSION_EVENT = {
    HIDE_AUX_HUD: "hide_aux_hud",
    TOGGLE_AUX_HUD: "toggleAuxHud",
    AUX_HUD_HIDDEN: "auxHudHidden",
    AUX_HUD_ESCAPED: "auxHudEscaped",
    ELEMENT_DIMENSION_CHANGED: "elementDimensionChanged",
    SELECTION_CHANGE: "dw_selection_changed",
    STYLES_RELOADED: "liveview_styles_reloaded",
    TEXT_EDIT_BEGIN: "text editing started",
    TEXT_EDIT_END: "text editing ended",
	TEXTEDIT_SELECTION_CHANGED: "Selection changed inside textedit",
    TEXTEDIT_SELECTION_LOST: "Selection lost inside textedit",
    SHOW_AUX_HUD: "show_aux_hud",
    RELOAD_ESH: "reload_esh",
    FILE_SAVE: "file_save",
    FILE_SAVE_AS: "file_save_as",
    VIEW_LOST_FOCUS: "liveview_lostfocus"
};

var DWLE_DIMENSIONS = {
    ImgHudWidth: 210,
    ImgHudHeight: 135,
    ElementHudDefault: 20,
    AuxHudOffSet: 13
};

DwExtensionEvent.prototype = {
/*
Event handling mechanisms for all HUDS.
Every HUD will hold instance of this object
Events List:
Please refer to Constants.JS and all the events are
listed under DW_EXTENSION_EVENT
*/
    constructor: DwExtensionEvent,

    addListener: function (type, listener, liveEditContext) {

        'use strict';
        if (!this.listeners[type]) {
            this.listeners[type] = [];
        }

        if (!this.contexts[type]) {
            this.contexts[type] = [];
        }

        this.listeners[type].push(listener);
        this.contexts[type].push(liveEditContext);
    },

    fire: function (event) {
        'use strict';
        var listeners, contexts, i, len;
        if (typeof event === "string") {
            event = {
                type: event
            };
        }
        if (!event.target) {
            event.target = this;
        }

        if (!event.type) {
            throw new Error("Dw Extension Event object missing 'type' property.");
        }

        if (this.listeners[event.type] instanceof Array) {
            listeners = this.listeners[event.type];
            contexts = this.contexts[event.type];
            for (i = 0, len = listeners.length; i < len; i += 1) {
                listeners[i].call(contexts[i], event);
            }
        }
    },

    removeListener: function (type, listener) {
        'use strict';
        var listeners, contexts, i, len;
        if (this.listeners[type] instanceof Array) {
            listeners = this.listeners[type];
            contexts = this.contexts[type];
            for (i = 0, len = listeners.length; i < len; i += 1) {
                if (listeners[i] === listener) {
                    listeners.splice(i, 1);
                    contexts.splice(i, 1);
                    break;
                }
            }
        }
    },
    
    messageHandler: function (evt) {
        'use strict';
        if (evt.data.dwExtensionEvent === true) {
            this.fire(evt.data);
        }
    },
    
    /**
     *  Sends a message to parent window in the Editable Live View
     *  @param message that is to be sent
     *  @return none
     */
    sendMessage: function (message) {
        'use strict';
        message.dwExtensionEvent = true;
//        alert("sending message to parent: " + message.type);
        window.parent.postMessage(message, "*");
    }
};

var dwExtensionController = new DwExtensionEvent();
window.dwExtensionController = dwExtensionController;
window.addEventListener("message", dwExtensionController.messageHandler.bind(dwExtensionController), false);
